/*******************************************************************************
 *
 *  HAL_AppUart.c - Code for controlling Application UART
 *
 *  Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/

/***************************************************************************//**
 * @file       board_spi.c
 * @addtogroup board_spi
 * @{
 ******************************************************************************/

#include "hal_rfspi.h"
#include "board_spi.h"
#include "hal_board.h"

/* API related includes */
#include "hci.h" 
#include "spi.h"

#if defined(__TI_COMPILER_VERSION__)
#define  BYTE_PACK __attribute__ ((packed, __aligned__(1)))
#else
#define  BYTE_PACK
#endif

#define TEST_HEADER_SIZE 	5
/* SPI Related Defines */
static volatile uint8_t gCallSpiIntHandler = 0;
uint8_t wlan_rx_buffer[SPI_BUFFER_SIZE] ={0xDE};
uint8_t wlan_tx_buffer[SPI_BUFFER_SIZE] = {0xDE};
/* Start user code for global. Do not edit comment generated here */

void CSI00_InterruptDisable(void);

void SpiWriteDataSynchronous(uint8_t *data, unsigned short size);
void SpiReadWriteString(uint8_t ulTrueFalse, uint8_t *ptrData,
		uint16_t ulDataSize);
void SpiReadWriteStringInt(uint8_t ulTrueFalse, uint8_t *ptrData,
		uint16_t ulDataSize);
void SpiIntGPIOHandler(void);

#define 	eSPI_STATE_POWERUP 				 (0)
#define 	eSPI_STATE_INITIALIZED  		 (1)
#define 	eSPI_STATE_IDLE					 (2)
#define 	eSPI_STATE_WRITE_IRQ	   		 (3)
#define 	eSPI_STATE_WRITE_FIRST_PORTION   (4)
#define 	eSPI_STATE_WRITE_EOT			 (5)
#define 	eSPI_STATE_READ_IRQ				 (6)
#define 	eSPI_STATE_READ_FIRST_PORTION	 (7)
#define 	eSPI_STATE_READ_EOT				 (8)

#define FWT_DELAY               4000
#define FIFTY_US_DELAY			((16*50)/5) // for 16 MHz clk
#define READ                    3
#define WRITE                   1

#define HI(value)               (((value) & 0xFF00) >> 8)
#define LO(value)               ((value) & 0x00FF)

#define HEADERS_SIZE_EVNT       (SPI_HEADER_SIZE + 5)

#define ASSERT_CS()          { halRfSpiCSLow(); }

#define DEASSERT_CS()        { halRfSpiCSHigh();}

typedef struct {
	gcSpiHandleRx SPIRxHandler;
	unsigned short usTxPacketLength;
	unsigned short usRxPacketLength;
	volatile unsigned long ulSpiState;
	unsigned char *pTxPacket;
	unsigned char *pRxPacket;

}BYTE_PACK tSpiInformation;

#if defined(__TI_COMPILER_VERSION__)
#else
#pragma pack(1)
#endif
typedef struct _btspi_hdr {
	unsigned char cmd;
	unsigned short length;
	unsigned char pad[2];
}BYTE_PACK btspi_hdr;


tSpiInformation sSpiInformation;
//
// Static buffer for 5 bytes of SPI HEADER
//
#if defined(__TI_COMPILER_VERSION__)
#pragma DATA_ALIGN(tSpiReadHeader,1)
#endif
uint8_t tSpiReadHeader[] = { READ, 0, 0, 0, 0 };
/* This buffer is used when we are sending data on SPI but we are not interested
 in reading from Slave*/
#if defined(__TI_COMPILER_VERSION__)
#pragma DATA_ALIGN(chBuffer,1)
#endif
unsigned char chBuffer[SPI_BUFFER_SIZE];
/* Declare local function definitions */

/* Start user code for adding. Do not edit comment generated here */

/**
 * @brief  The functions delay for a number of MCU clk period
 * @param  The number of MCU clk period to delay for
 * @retval None
 */
void SysCtlDelay(unsigned long Delay) {
	/* Decrement nCount value */
	unsigned long ulDelay = Delay;
	while (ulDelay) {
		ulDelay--;
	}
}

/**
 * @brief  Opens the Spi Port and specify and ISR Handler
 * @param  Data: byte to send.
 * @retval None
 */
void SpiOpen(gcSpiHandleRx pfRxHandler) {

	memset(wlan_tx_buffer, 0xE3, SPI_BUFFER_SIZE);
	memset(wlan_rx_buffer, 0xE3, SPI_BUFFER_SIZE);
	sSpiInformation.ulSpiState = eSPI_STATE_POWERUP;
	sSpiInformation.SPIRxHandler = pfRxHandler;

	sSpiInformation.pRxPacket = wlan_rx_buffer;
	sSpiInformation.usRxPacketLength = 0;
	sSpiInformation.usTxPacketLength = 0;
	//
	// Enable interrupt on the GPIOA pin of WLAN IRQ
	//
	tSLInformation.WlanInterruptEnable();
}

/**
 * @brief  Opens the Spi Port and specify and ISR Handler
 * @param  Data: byte to send.
 * @retval None
 */
void SpiClose(void) {
	if (sSpiInformation.pRxPacket) {
		sSpiInformation.pRxPacket = 0;
	}
	//
	//	Disable Interrupt in GPIOA module...
	//
	tSLInformation.WlanInterruptDisable();
}

/**
 * @brief  Sends data on SPI to generate interrupt on reception
 * @param  The pointer to data buffer
 * @param  This size of data
 * @retval None
 */
void SpiReadData(unsigned char *data, unsigned short size) {
	SpiReadWriteString(TRUE, data, size);
}

/**
 * @brief  A wrapper to enable interrupt on IRQ line
 * @param  None
 * @retval None
 */
void SpiResumeSpi(void) {
	//
	//	Enable Interrupt on WLAN IRQ Pin
	//
	tSLInformation.WlanInterruptEnable();
}

void SpiPauseSpi(void) {
	//
	//	Enable Interrupt on WLAN IRQ Pin
	//
	tSLInformation.WlanInterruptDisable();
}
/**
 * @brief  This indicate the end of a receive and calls a registered handler
 to process the received data
 * @param  None
 * @retval None
 */
void SpiTriggerRxProcessing(void) {
	//
	// Trigger Rx processing
	//
	SpiPauseSpi();
	DEASSERT_CS();

	sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
	sSpiInformation.SPIRxHandler(
			sSpiInformation.pRxPacket + sizeof(btspi_hdr));
}
/**
 * @brief  This sends data over the SPI transport layer with
 * @param  None
 * @retval None
 */
void SpiWriteAsync(unsigned char *data, unsigned short size) {
	SpiReadWriteString(FALSE, data, size);
}

/**
 * @brief  This function TX and RX SPI data with no interrupt at end of SPI TX
 * @param  ulTrueFlase True for a read or False for write
 * @param  ptrData Pointer to data to be written
 * @param  ulDataSize The size of the data to be written or read
 * @retval None
 */
void SpiReadWriteString(uint8_t ulTrueFalse, uint8_t *ptrData,
		uint16_t ulDataSize) {
	SysCtlDelay(FIFTY_US_DELAY);
	if (ulTrueFalse == TRUE)
	{
		halRFSpiReadWriteFrame((uint8_t*) tSpiReadHeader, (uint8_t*) ptrData,
				ulDataSize);
	} else {
		halRFSpiReadWriteFrame((uint8_t*) ptrData,
				(uint8_t*) sSpiInformation.pRxPacket, ulDataSize);
	}
}

/**
 * @brief  This function TX and RX SPI data with no interrupt at end of SPI TX
 * @param  ptrData Pointer to data to be written
 * @param  ulDataSize The size of the data to be written or read
 * @retval None
 */
void SpiWriteDataSynchronous(uint8_t *data, unsigned short size) {
	SpiReadWriteString(TRUE, data, size);
}
/**
 * @brief  Sends header information to CC3000
 * @param  None
 * @retval None
 */
static long SpiFirstWrite(unsigned char *ucBuf, unsigned short usLength) {
	//
	// workaround for first transaction
	//
	ASSERT_CS();
	//SPI writes first 4 bytes of data
	SpiReadWriteString(FALSE, ucBuf, 4);
	//SPI writes next 4 bytes of data
	SpiReadWriteString(FALSE, ucBuf + 4, usLength - 4);
	// From this point on - operate in a regular way
	sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
	DEASSERT_CS();
	return (0);
}

/**
 * @brief  Writes data over SPI  transport link to CC3000
 * @param  pUserBuffer: pointer to data
 * @param usLength: length of data that will be sent to CC3000
 * @retval None
 */
long SpiWrite(unsigned char *pUserBuffer, unsigned short usLength) {
	unsigned char ucPad = 0;
	//
	// Figure out the total length of the packet in order to figure out if there is padding or not
	//
	if (!(usLength & 0x0001)) {
		ucPad++;
	}

	pUserBuffer[0] = WRITE;
	pUserBuffer[1] = HI(usLength + ucPad);
	pUserBuffer[2] = LO(usLength + ucPad);
	pUserBuffer[3] = 0;
	pUserBuffer[4] = 0;

	usLength += (sizeof(btspi_hdr) + ucPad);

	if (sSpiInformation.ulSpiState == eSPI_STATE_POWERUP)
	{
		while (sSpiInformation.ulSpiState != eSPI_STATE_INITIALIZED)
			;
	}
	if (sSpiInformation.ulSpiState == eSPI_STATE_INITIALIZED)
	{

		//
		// This is time for first TX/RX transactions over SPI: the IRQ is down - so need to send read buffer size command
		//
		SpiFirstWrite(pUserBuffer, usLength);
	} else {

		//
		// We need to prevent here race that can occur in case 2 back to back packets are sent to the 
		// device, so the state will move to IDLE and once again to not IDLE due to IRQ
		//
		tSLInformation.WlanInterruptDisable();
		while (sSpiInformation.ulSpiState != eSPI_STATE_IDLE) {
			;
		}
		sSpiInformation.ulSpiState = eSPI_STATE_WRITE_IRQ;
		sSpiInformation.pTxPacket = pUserBuffer;
		sSpiInformation.usTxPacketLength = usLength;

		//
		// Assert the CS line and wait till SSI IRQ line is active and then initialize write operation
		//
		tSLInformation.WlanInterruptEnable();

		ASSERT_CS();
	}
	// Due to the fact that we are currently implementing a blocking situation
	// here we will wait till end of transaction
	//	
	while (eSPI_STATE_IDLE != sSpiInformation.ulSpiState) {
		;
	}
	return (0);
}

/**
 * @brief  This function processes received SPI Header and in accordance with it
 - continues reading the packet

 * @param  None
 * @retval None
 */
long SpiReadDataCont(void) {
	hci_hdr_t *hci_hdr;
	long data_to_recv;
	hci_evnt_hdr_t *hci_evnt_hdr;
	unsigned char *evnt_buff;
	hci_data_hdr_t *pDataHdr;
	//
	//determine what type of packet we have
	//
	evnt_buff = sSpiInformation.pRxPacket;
	data_to_recv = 0;
	hci_hdr = (hci_hdr_t *) (evnt_buff + sizeof(btspi_hdr));

	switch (hci_hdr->ucType) {
	case HCI_TYPE_DATA: {
		pDataHdr = (hci_data_hdr_t *) (evnt_buff + sizeof(btspi_hdr));

		//
		// We need to read the rest of data..
		//
		data_to_recv = pDataHdr->usLength;

		if (!((HEADERS_SIZE_EVNT + data_to_recv) & 1)) {
			data_to_recv++;
		}

		if (data_to_recv) {
			SpiReadWriteString(TRUE, evnt_buff + 10, data_to_recv);
		}
		break;
	}
	case HCI_TYPE_EVNT: {
		//
		//configure buffer to read rest of the data
		//
		hci_evnt_hdr = (hci_evnt_hdr_t *) hci_hdr;

		//
		// Calculate the rest length of the data
		//
		data_to_recv = hci_evnt_hdr->ucLength - 1;

		//
		// Add padding byte if needed
		//
		if ((HEADERS_SIZE_EVNT + data_to_recv) & 1) {

			data_to_recv++;
		}

		if (data_to_recv) {
			SpiReadWriteString(TRUE, evnt_buff + 10, data_to_recv);
		}

		sSpiInformation.ulSpiState = eSPI_STATE_READ_EOT;
		break;
	}
	}

	return (0);
}

/**
 * @brief  This function enter point for read flow: first we read minimal
 5 SPI header bytes and 5 Event Data bytes

 * @param  None
 * @retval None
 */
void SpiReadHeader(void) {
	sSpiInformation.ulSpiState = eSPI_STATE_READ_IRQ;
	SpiReadWriteString(TRUE, sSpiInformation.pRxPacket, 10);
}

/**
 * @brief  Determine if all data was read if so end the data exchange
 * @param  None
 * @retval None
 */
void SSIContReadOperation(void) {
	//
	// The header was read - continue with  the payload read
	//
	if (!SpiReadDataCont()) {
		//
		// All the data was read - finalize handling by switching to teh task
		//	and calling from task Event Handler
		//
		SpiTriggerRxProcessing();
	}
}

/**
 * @brief  The handler for Interrupt that is generated when CC3000 brings the
 IRQ line low.
 * @param  None
 * @retval None
 */
void SpiIntGPIOHandler(void) {
	//Flag is cleared in first ISR handler

	if (!tSLInformation.ReadWlanInterruptPin()) {

		if (sSpiInformation.ulSpiState == eSPI_STATE_POWERUP)
		{
			/* This means IRQ line was low call a callback of HCI Layer to inform on event */
			sSpiInformation.ulSpiState = eSPI_STATE_INITIALIZED;
		} else if (sSpiInformation.ulSpiState == eSPI_STATE_IDLE)
		{

			sSpiInformation.ulSpiState = eSPI_STATE_READ_IRQ;
			/* IRQ line goes down - we are starting reception */
			ASSERT_CS();
			//
			// Wait for TX/RX Complete which will come as DMA interrupt
			//
			SpiReadHeader();
			sSpiInformation.ulSpiState = eSPI_STATE_READ_EOT;

			//
			//
			//
			SSIContReadOperation();
		} else if (sSpiInformation.ulSpiState == eSPI_STATE_WRITE_IRQ)
		{

			SpiReadWriteString(FALSE, sSpiInformation.pTxPacket,
					sSpiInformation.usTxPacketLength);
			sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
			DEASSERT_CS();
		}
	}
}

/* End user code. Do not edit comment generated here */
